export interface EmitableEvent {
    [event: string]: (...params: any) => any;
}
export interface Listener<T extends (...params: any) => any> {
    fn: T;
    once?: boolean;
    immediate?: boolean;
}
export interface ListenerOptions {
    once: boolean;
    immediate: boolean;
}
type EmitFn<F extends (...params: any) => any> = (events: Listener<F>[], ...params: Parameters<F>) => any;
export declare class EventEmitter<T extends EmitableEvent = {}> {
    protected events: {
        [P in keyof T]?: Listener<T[P]>[];
    };
    private emitted;
    protected emitter: {
        [P in keyof T]?: EmitFn<T[P]>;
    };
    /**
     * 监听某个事件
     * @param event 要监听的事件类型
     * @param fn 触发事件时执行的函数
     * @param options 监听选项
     */
    on<K extends keyof T>(event: K, fn: T[K], options?: Partial<ListenerOptions>): void;
    /**
     * 取消监听某个事件
     * @param event 要取消监听的事件类型
     * @param fn 要取消监听的函数
     */
    off<K extends keyof T>(event: K, fn: T[K]): void;
    /**
     * 监听事件，并只触发一次
     * @param event 要监听的事件
     * @param fn 监听函数
     */
    once<K extends keyof T>(event: K, fn: T[K]): void;
    /**
     * 触发某个事件
     * @param event 要触发的事件类型
     * @param params 传入的参数
     */
    emit<K extends keyof T>(event: K, ...params: Parameters<T[K]>): ReturnType<T[K]>[];
    emit<K extends keyof T, R>(event: K, ...params: Parameters<T[K]>): R;
    /**
     * 设置一个事件的执行器(emitter)
     * @param event 要设置的事件
     * @param fn 事件执行器，留空以清除触发器
     */
    setEmitter<K extends keyof T>(event: K, fn?: EmitFn<T[K]>): void;
    /**
     * 取消监听所有的事件，删除所有监听函数
     */
    removeAllListeners(): void;
    /**
     * 取消监听一个事件的所有函数
     * @param event 要取消监听的事件
     */
    removeAllListeners(event: keyof T): void;
}
type IndexedSymbol = number | string | symbol;
export declare class IndexedEventEmitter<T extends EmitableEvent> extends EventEmitter<T> {
    private fnMap;
    /**
     * 监听事件，并将函数与唯一标识符绑定
     * @param event 事件类型
     * @param symbol 监听函数的唯一标识符
     * @param fn 监听函数
     * @param options 监听配置
     */
    onIndex<K extends keyof T>(event: K, symbol: IndexedSymbol, fn: T[K], options?: Partial<ListenerOptions>): void;
    /**
     * 监听事件，绑定唯一标识符，但监听函数只会执行一次
     * @param event 要监听的事件
     * @param symbol 监听函数的唯一标识符
     * @param fn 监听函数
     */
    onceIndex<K extends keyof T>(event: K, symbol: IndexedSymbol, fn: T[K]): void;
    /**
     * 取消监听一个事件
     * @param event 要取消监听的事件
     * @param symbol 取消监听的函数的唯一标识符
     */
    offIndex<K extends keyof T>(event: K, symbol: IndexedSymbol): void;
    private ensureMap;
}
export {};
