import { Component } from 'vue';
import { EmitableEvent, EventEmitter } from '../../common/eventEmitter';
import { Hotkey } from './hotkey';
interface FocusEvent<T> extends EmitableEvent {
    focus: (before: T | null, after: T) => void;
    unfocus: (before: T | null) => void;
    add: (item: T) => void;
    pop: (item: T | null) => void;
    splice: (spliced: T[]) => void;
}
export declare class Focus<T = any> extends EventEmitter<FocusEvent<T>> {
    /** 显示列表 */
    stack: T[];
    focused: T | null;
    /** 聚焦目标是否平等，在平等时，关闭聚焦目标不再会将其之后的目标全部删除，而是保留 */
    readonly equal: boolean;
    constructor(react?: boolean, equal?: boolean);
    /**
     * 聚焦于一个目标
     * @param target 聚焦目标
     * @param add 如果聚焦目标不在显示列表里面，是否自动追加
     */
    focus(target: T, add?: boolean): void;
    /**
     * 取消聚焦
     */
    unfocus(): void;
    /**
     * 向显示列表中添加物品
     * @param item 添加的物品
     */
    add(item: T): void;
    /**
     * 弹出显示列表中的最后一个物品
     */
    pop(): NonNullable<T> | null;
    /**
     * 从一个位置开始删除显示列表，如果平等，则只会删除一个，否则会将其之后的所有的目标全部删除
     * @param item 从哪开始删除，包括此项
     */
    splice(item: T): void;
    /**
     * 根据索引在显示列表中删除一项
     * @param index 要删除的项的索引
     */
    spliceIndex(index: number): void;
}
interface GameUiEvent extends EmitableEvent {
    close: () => void;
    open: () => void;
}
interface ShowableGameUi {
    ui: GameUi;
    vOn?: UiVOn;
    vBind?: UiVBind;
}
type UiVOn = Record<string, (param?: any) => void>;
type UiVBind = Record<string, any>;
interface MountedVBind {
    num: number;
    ui: GameUi;
    [x: string]: any;
}
export declare class GameUi extends EventEmitter<GameUiEvent> {
    static uiList: GameUi[];
    component: Component;
    hotkey?: Hotkey;
    id: string;
    symbol: symbol;
    constructor(id: string, component: Component, hotkey?: Hotkey);
    /**
     * 根据 v-on 和 v-bind 创建可以显示的 ui 组件
     * @param vOn 监听的事件
     * @param vBind 绑定的数据
     */
    with(vBind?: UiVBind, vOn?: UiVOn): ShowableGameUi;
}
interface IndexedGameUi extends ShowableGameUi {
    num: number;
    vBind?: MountedVBind;
}
interface HoldOnController {
    end(noClosePanel?: boolean): void;
}
export declare class UiController extends Focus<IndexedGameUi> {
    static list: UiController[];
    list: Record<string, GameUi>;
    num: number;
    show: 'end' | 'all';
    private hold;
    constructor(equal?: boolean);
    /**
     * 设置为仅显示最后一个ui
     */
    showEnd(): void;
    /**
     * 设置为显示所有ui
     */
    showAll(): void;
    /**
     * 根据id获取到ui
     * @param id ui的id
     */
    get(id: string): GameUi;
    /**
     * 暂时保持下一次删除ui不会导致ui整体被关闭，引起ui背景闪烁。
     * 例如可以用于道具栏，打开道具时就应当 holdOn，然后通过道具使用钩子来判断接下来是否要隐藏 app:
     * ```txt
     * hold on -> close -> use item -> hook -> stack.length === 0 ? end() : no action
     * ```
     */
    holdOn(): HoldOnController;
    /**
     * 关闭一个ui，注意如果不是平等模式，在其之后的ui都会同时关闭掉
     * @param num 要关闭的ui的唯一标识符
     */
    close(num: number): void;
    /**
     * 根据id关闭所有同id的ui，注意非平等模式下，会将第一个ui后的所有ui都关闭掉
     * @param id 要关闭的ui的id
     */
    closeByName(id: string): void;
    /**
     * 打开一个新的ui
     * @param id 要打开的ui的id
     * @param vOn 监听的事件
     * @param vBind 绑定的数据
     * @returns ui的唯一标识符
     */
    open(id: string, vBind?: UiVBind, vOn?: UiVOn): number;
    /**
     * 注册一个ui
     * @param id ui的id
     * @param ui 对应的GameUi实例
     */
    register(...ui: GameUi[]): void;
    /**
     * 取消注册一个ui
     * @param id 要取消注册的ui的id
     */
    unregister(...id: string[]): this;
    /**
     * 根据ui的唯一标识符进行聚焦
     * @param num 要聚焦于的ui的唯一标识符
     */
    focusByNum(num: number): void;
    /**
     * 根据唯一标识符获取对应的ui
     * @param num ui的唯一标识符
     */
    getByNum(num: number): IndexedGameUi | undefined;
    /**
     * 根据ui的唯一标识符来判断当前是否存在某个ui
     * @param id ui的唯一标识符
     */
    hasName(id: string): boolean;
}
export {};
