import { TimingFn } from 'mutate-animate';
interface AnimatedPathShadow {
    offsetX: number | TimingFn;
    offsetY: number | TimingFn;
    blur: number | TimingFn;
    color: string | TimingFn<4>;
}
type AnimatedPathFilterKey = 'blur' | 'brightness' | 'contrast' | 'grayscale' | 'hueRotate' | 'opacity' | 'saturate' | 'sepia';
type AnimatedPathFilter = Record<AnimatedPathFilterKey, number | TimingFn>;
interface Path {
    path: TimingFn<2>;
    length: number;
}
export default function init(): {
    AnimatedPath: typeof AnimatedPath;
    pathTest: typeof test;
};
export declare class AnimatedPath {
    /** 不同线条间是否连接起来，不连接的话中间可能会有短暂的中断 */
    join: boolean;
    /** 路径信息 */
    linePath: Path[];
    /** 绘制画布 */
    ctx: CanvasRenderingContext2D;
    private dashStatus;
    private dashLength;
    private dashMode;
    private lineWidth;
    private lineColor;
    private lineShadow;
    private lineFilter;
    private pathClose;
    constructor(ctx: CanvasRenderingContext2D);
    /**
     * 设置虚线格式
     * @param mode 一个偶数个元素的数组，表示虚线格式，如果不是偶数个，那么会再复制一份加到后面。
     * 元素可以填数字或一个函数，如果是数字，表示该值是定值，
     * 如果是函数，该函数包括一个参数input，表示绘制动画的完成度，输出一个数字，表示虚线的长度。
     * @example path.dash([5, 10]); //  表示绘制时，会先绘制5像素的实线，之后10像素不绘制，然后再绘制5像素实线，以此类推。
     * @example path.dash([5, (input) => Math.round(input * 10)]);
     * // 表示绘制时，先绘制5像素的实线，然后会有一段不绘制，不绘制的长度是动画完成度乘10，以此类推。
     */
    dash(mode: (number | TimingFn)[]): this;
    /**
     * 设置线条宽度
     * @param width 绘制的线条宽度，如果是数字，表示宽度是个常量。
     * 也可以传入一个函数，函数包括一个参数input，表示绘制动画的完成度，输出一个数字，表示宽度。
     * @example path.width(2); // 设置线条宽度为2像素
     * @example path.width((input) => input * 5); // 设置线条宽度为动画完成度的5倍
     */
    width(width: number | TimingFn): this;
    /**
     * 设置线条颜色
     * @param color 颜色，可以传入css颜色字符串或函数，传入字符串表示线条颜色为定值，
     * 如果传入函数，这个函数有一个参数input，表示动画完成度，输出一个有4个元素的数组，表示颜色的rgba值。
     * @example path.color('#fff'); // 设置线条为白色
     * @example path.color((input) => [input * 100, input * 50, input * 255, input * 0.8]);
     * // 设置颜色的红色值为动画完成度的100倍，绿色为动画完成度的50倍，蓝色为动画完成度的255倍，不透明度为动画完成度的0.8倍
     */
    color(color: string | TimingFn<4>): this;
    /**
     * 设置线条阴影
     * @param shadow 阴影信息，一个对象，包含offsetX(横向偏移量), offsetY(纵向偏移量), blur(虚化程度), color(颜色)
     * 四个属性，均为可选，前三个可传入数字或函数，color可传入字符串或函数，传入的函数与前面几个方法中的函数类似。
     * ```ts
     * path.shadow({
     *     offsetX: 3, // 横向偏移量为3
     *     offsetY: input => input * 10, // 纵向偏移量为动画完成度的10倍
     *     color: '#0008', // 颜色为半透明的黑色
     *     blur: 4 // 虚化程度为4
     * })
     * ```
     */
    shadow(shadow: Partial<AnimatedPathShadow>): this;
    /**
     * 设置线条的滤镜
     * @param filter 滤镜信息，一个对象，可以有以下属性：
     * 1. `blur`: 虚化程度
     * 2. `brightness`: 亮度，百分比，`0-Infinity`
     * 3. `contrast`: 对比度，百分比，`0-Infinity`
     * 4. `grayscale`: 黑白度，百分比，`0-100`
     * 5. `hueRotate`: 色相旋转，角度，`0-360`
     * 6. `invert`: 反色，百分比，`0-100`
     * 7. `opacity`: 不透明度，百分比，`0-100`
     * 8. `saturate`: 饱和度，百分比，`0-Infinity`
     * 9. `sepia`: 深褐色(怀旧风格)，百分比，`0-100`
     * 以上属性均可选，均可传入数字或函数。
     * ```ts
     * path.filter({
     *     blur: 3, // 虚化程度为3
     *     contrast: input => 100 + input * 50 // 对比度增加动画完成度的50倍
     * })
     * ```
     */
    filter(filter: Partial<AnimatedPathFilter>): this;
    /**
     * 清空路径
     */
    clear(): this;
    /**
     * 添加直线
     * @param x1 起点横坐标
     * @param y1 起点纵坐标
     * @param x2 终点横坐标
     * @param y2 终点纵坐标
     * @returns
     */
    line(x1: number, y1: number, x2: number, y2: number): this;
    /**
     * 添加圆弧
     * @param x 圆心横坐标
     * @param y 圆心纵坐标
     * @param r 圆半径
     * @param start 圆起始角度，弧度制，水平向右表示0弧度，顺时针旋转
     * @param end 圆终止角度，弧度制
     */
    circle(x: number, y: number, r: number, start: number, end: number): this;
    /**
     * 添加一个椭圆
     * @param x 圆心横坐标
     * @param y 圆心纵坐标
     * @param a 椭圆横轴长
     * @param b 椭圆纵轴长
     * @param start 起始角度，弧度制
     * @param end 终止角度，弧度制
     */
    ellipse(x: number, y: number, a: number, b: number, start: number, end: number): this;
    /**
     * 添加一个矩形
     * @param x 左上角横坐标
     * @param y 左上角纵坐标
     * @param w 宽度
     * @param h 高度
     * @param lineWidth 线宽，用于闭合矩形
     */
    rect(x: number, y: number, w: number, h: number, lineWidth?: number): this;
    /**
     * 添加一个贝塞尔曲线
     * @param point 起点，控制点，终点的坐标
     */
    bezier(...point: [number, number][]): this;
    /**
     * 添加一个路径
     * @param path 路径函数，输入一个绘制完成度，输出一个坐标
     * @param length 路径的长度，不填则会调用calLength进行计算，用于该段路径的绘制时间的计算，
     * 也可以通过设置这个值来指定该段路径的绘制时间。请尽量指定该值，不然的话性能表现会较差，如果大量添加路径可能会导致卡顿。
     * @example path.add(input => [input * 100, (input * 100) ** 2]); // 添加一个抛物线路径
     */
    add(path: TimingFn<2>, length?: number): this;
    /**
     * 设置路径是否闭合
     * @param close 路径是否闭合
     */
    close(close: boolean): this;
    /**
     * 对一个路径进行线积分，计算其长度。注意该函数性能较差，请谨慎使用
     * @param path 路径函数，输入一个绘制完成度，输出一个坐标
     * @returns 路径的长度
     */
    calLength(path: TimingFn<2>): number;
    /**
     * 立即执行绘制，不执行动画，该函数的性能表现可能较差，请谨慎使用
     */
    drawImmediate(): this;
    /**
     * 执行绘制
     * @param time 绘制时间，如果是0则会直接调用drawImmediate函数，注意如果时间太短的话可能会造成明显的锯齿现象
     * @param timing 绘制完成度的速率函数，输入一个时间完成度，输出一个绘制完成度
     */
    draw(time: number, timing?: TimingFn): this;
    private drawFrom;
    private handleFx;
}
declare function test(): Promise<void>;
export {};
