import { TimingFn } from 'mutate-animate';
import { Matrix4 } from '../webgl/matrix';
import { Renderer } from './render';
export type Position3D = [number, number, number];
type OneParameterAnimationType = 'x' | 'y' | 'z';
type TwoParameterAnimationType = 'xy' | 'yx' | 'xz' | 'zx' | 'yz' | 'zy';
type ThreeParameterAnimationType = 'xyz' | 'xzy' | 'yxz' | 'yzx' | 'zxy' | 'zyx';
type CameraAnimationTarget = 'eye' | 'at' | 'up';
interface CameraAnimationType {
    1: OneParameterAnimationType;
    2: TwoParameterAnimationType;
    3: ThreeParameterAnimationType;
}
export declare class Camera {
    /** 视图矩阵 */
    view: Matrix4;
    /** 投影矩阵 */
    projection: Matrix4;
    /** 绑定的渲染器 */
    renderer?: Renderer;
    constructor();
    /**
     * 初始化视角矩阵
     */
    reset(): void;
    /**
     * 将该摄像机与一个渲染器绑定
     * @param renderer 渲染器
     */
    bind(renderer: Renderer): void;
    /**
     * 取消与渲染器的绑定
     */
    unbind(): void;
    /**
     * 设置摄像机的观察方位
     * @param eye 视点位置
     * @param at 目标点位置
     * @param up 上方向
     */
    lookAt(eye: Position3D, at: Position3D, up: Position3D): void;
    /**
     * 变换摄像机的观察方位
     * @param eye 视点位置
     * @param at 目标点位置
     * @param up 上方向
     */
    transform(eye: Position3D, at: Position3D, up: Position3D): void;
    /**
     * 设置透视投影矩阵
     * @param fov 垂直视角，即摄像机视锥体的上下平面夹角，单位角度
     * @param aspect 近裁剪面的长宽比，即视野的长宽比
     * @param near 近裁剪面的距离，即最近能看多远
     * @param far 远裁剪面的距离，即最远能看多远
     */
    setPerspective(fov: number, aspect: number, near: number, far: number): void;
    /**
     * 设置正交投影矩阵
     * @param left 可视空间的左边界
     * @param right 可视空间的右边界
     * @param bottom 可视空间的下边界
     * @param top 可视空间的上边界
     * @param near 近裁剪面的距离，即最近能看多远
     * @param far 远裁剪面的距离，即最远能看多远
     */
    setOrthogonal(left: number, right: number, bottom: number, top: number, near: number, far: number): void;
    /**
     * 更新视角
     */
    update(): void;
    applyAnimate<N extends keyof CameraAnimationType = 1>(target: CameraAnimationTarget, type: CameraAnimationType[N], time?: number, timing?: TimingFn<N>, relative?: boolean): void;
    /**
     * 计算摄像机变换矩阵
     * @see https://github.com/bad4iz/cuon-matrix/blob/main/src/Matrix4/Matrix4.ts
     * @param eye 视点位置
     * @param at 目标点位置
     * @param up 上方向
     * @returns 转换矩阵
     */
    private calLookAt;
    /**
     * 计算透视矩阵
     * @see https://github.com/bad4iz/cuon-matrix/blob/main/src/Matrix4/Matrix4.ts
     * @param fovy 垂直视角，即摄像机视锥体的上下平面夹角
     * @param aspect 近裁剪面的长宽比，即视野的长宽比
     * @param near 近裁剪面的距离，即最近能看多远
     * @param far 远裁剪面的距离，即最远能看多远
     */
    private calPerspective;
    /**
     * 设置正交投影矩阵
     * @param left 可视空间的左边界
     * @param right 可视空间的右边界
     * @param bottom 可视空间的下边界
     * @param top 可视空间的上边界
     * @param near 近裁剪面的距离，即最近能看多远
     * @param far 远裁剪面的距离，即最远能看多远
     */
    private calOrthogonal;
}
export {};
