import { Ticker } from 'mutate-animate';
import { EmitableEvent, EventEmitter } from '../common/eventEmitter';
interface ShaderEvent extends EmitableEvent {
}
type ShaderColorArray = [number, number, number, number];
type ShaderEffectImage = Exclude<TexImageSource, VideoFrame | ImageData>;
interface ProgramInfo {
    program: WebGLProgram;
    attrib: Record<string, number>;
    uniform: Record<string, WebGLUniformLocation>;
}
interface ShaderEffectBuffer {
    position: WebGLBuffer;
    texture: WebGLBuffer;
    indices: WebGLBuffer;
}
interface ShaderEffectShader {
    vertex: WebGLShader;
    fragment: WebGLShader;
}
interface MixedImage {
    canvas: HTMLCanvasElement;
    update(): void;
}
type UniformBinderNum = 1 | 2 | 3 | 4;
type UniformBinderType = 'f' | 'i';
type UniformBinderValue<N extends UniformBinderNum> = N extends 1 ? number : N extends 2 ? [number, number] : N extends 3 ? [number, number, number] : [number, number, number, number];
interface UniformBinder<N extends UniformBinderNum, T extends UniformBinderType, V extends 'v' | ''> {
    value: UniformBinderValue<N>;
    set(value: UniformBinderValue<N>): void;
    get(): UniformBinderValue<N>;
}
export declare class ShaderEffect extends EventEmitter<ShaderEvent> {
    canvas: HTMLCanvasElement;
    gl: WebGLRenderingContext;
    program: WebGLProgram | null;
    texture: WebGLTexture | null;
    programInfo: ProgramInfo | null;
    buffer: ShaderEffectBuffer | null;
    shader: ShaderEffectShader | null;
    textureCanvas: MixedImage | null;
    private baseImages;
    private background;
    private _vsSource;
    private _fsSource;
    get vsSource(): string;
    get fsSource(): string;
    constructor(background: ShaderColorArray);
    /**
     * 设置特效作用于的图片，不会修改原图片，而是会在 ShaderEffect.canvas 画布元素中展现
     * @param img 特效作用于的图片
     */
    baseImage(...img: ShaderEffectImage[]): void;
    /**
     * 强制重新渲染特效
     * @param compile 是否重新编译着色器脚本，并重新创建纹理
     */
    update(compile?: boolean): void;
    /**
     * 仅重新编译着色器，不进行纹理创建和特效渲染
     */
    compile(): void;
    /**
     * 设置顶点着色器，使用 glsl 编写，插件提供了一些新的 api
     * 着色器中必须包含 main 函数，同时为 gl_Position 赋值
     * @param shader 顶点着色器代码
     */
    vs(shader: string): void;
    /**
     * 设置片元着色器，使用 glsl 编写，插件提供了一些新的 api
     * 着色器中必须包含 main 函数，同时为 gl_FragColor 赋值
     * @param shader 片元着色器代码
     */
    fs(shader: string): void;
    /**
     * 绘制特效
     */
    drawScene(): void;
    /**
     * 创建一个全局变量绑定器，用于操作全局变量
     * @param uniform 全局变量的变量名
     * @param num 变量的元素数量，float和int视为1，vec2 vec3 vec4分别视为 2 3 4
     * @param type 数据类型，可以填'f'，表示浮点型，或者填'i'，表示整型
     * @param vector 是否为向量，可以填'v'，表示是向量，或者填''，表示不是向量
     * @returns 一个uniform绑定器，用于操作全局变量uniform
     */
    createUniformBinder<N extends UniformBinderNum, T extends UniformBinderType, V extends 'v' | ''>(uniform: string, num: N, type: T, vector: V): UniformBinder<N, T, V>;
    private createProgram;
    private loadShader;
    private createTexture;
    private initBuffers;
    private initBuffer;
    private initIndexBuffer;
    private getProgramInfo;
    private setTextureAttrib;
    private setPositionAttrib;
    static defaultVs: string;
    static defaultFs: string;
}
interface GameCanvasReplacer {
    recover(): void;
    append(): void;
    remove(): void;
    update(compile?: boolean): void;
}
/**
 * 为一个着色器特效创建每帧更新的 ticker，部分条件下性能表现可能较差
 * @param effect 要每帧更新的着色器特效
 */
export declare function setTickerFor(effect: ShaderEffect): Ticker;
/**
 * 用着色器特效画布替换样板画布
 * @param effect 着色器特效实例
 * @param canvas 要替换的画布列表
 * @returns 特效控制器，用于控制特效的显示
 */
export declare function replaceGameCanvas(effect: ShaderEffect, canvas: string[]): GameCanvasReplacer;
export {};
